/*
 * @(#)XslPattern.java	1.5 98/12/17
 * 
 * Copyright (c) 1998 Sun Microsystems, Inc. All Rights Reserved.
 * 
 * This software is the confidential and proprietary information of Sun
 * Microsystems, Inc. ("Confidential Information").  You shall not
 * disclose such Confidential Information and shall use it only in
 * accordance with the terms of the license agreement you entered into
 * with Sun.
 * 
 * SUN MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF THE
 * SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR
 * PURPOSE, OR NON-INFRINGEMENT. SUN SHALL NOT BE LIABLE FOR ANY DAMAGES
 * SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING OR DISTRIBUTING
 * THIS SOFTWARE OR ITS DERIVATIVES.
 */

package com.sun.xml.xsl;

import java.io.*;

import org.w3c.dom.Element;

import com.sun.xml.util.XmlChars;
import com.sun.xml.tree.ElementNode;


/**
 * Represent basic match patterns.
 */
abstract class XslPattern
{
    /**
     * Parses the match pattern string and returns an object
     * which will recognize conformant elements; or, throw an
     * IO Exception if the pattern can't be parsed.
     *
     * <P> For now, absolute/id anchors (the only kind allowed in
     * match patterns) aren't parsed.  The same is true for
     * element qualifiers (constraining children, attributes, or
     * positions).
     */
    static XslPattern parse (String pattern) throws IOException
    {
    	// remove all whitespace -- a quick'n'dirty hack, incorrect in
	// the (unsupported) case of attribute value qualifiers since
	// they can have internal spaces.

    	StringBuffer	buf = new StringBuffer ();
    	
    	for (int i = 0; i < pattern.length (); i++) {
    	    char c = pattern.charAt (i);
    	    if (c == ' ' || c == '\t' || c == '\n' || c == '\r')
    	        continue;
    	    buf.append (c);
    	}
    	pattern = buf.toString ();
    	
    	return new OrPattern (new StringReader (pattern));
    }
    
    // "match" patterns match an element
    // Style sheet is passed explicitly for ID mapping 
    abstract boolean match (Element e, StyleSheet s);
    
    // probably need an isMatchPattern predicate ... select
    // patterns allow "." and ".." anchors
    
    
    // [2] AncestryPattern ( '|' AncestryPattern )*
    static class OrPattern extends XslPattern
    {
    	private OrPattern	next;
    	private AncestryPattern	self;
    	
    	OrPattern (Reader r) throws IOException
    	{
    	    boolean	needNext = false;
    	    self = new AncestryPattern (r);
    	    try {
    	    	needNext = (r.read () == '|');
    	    	if (needNext)
    	    	    next = new OrPattern (r);
    	    } catch (IOException e) {
    	    	if (needNext)
    	    	    throw e;
    	    }
    	}
    	
    	boolean match (Element e, StyleSheet s)
    	{
    	    if (self.match (e, s))
    	        return true;
    	    else if (next != null)
    	        return next.match (e, s);
    	    else
    	        return false;
    	}
    }
    
    // [3] NodePatterns
    //		| (Anchor (AncestryOp NodePatterns)?)
    //		| (RootPattern NodePatterns?)
    static class AncestryPattern extends XslPattern
    {
    	private String		rootPattern;
    	private NodePatterns	nodePatterns;
    	private IdAnchor	anchor;
    	private String		ancestryOp;
    	
    	AncestryPattern (Reader r) throws IOException
    	{
    	    if ((nodePatterns = NodePatterns.get (r)) != null)
    	        return;
    	    if ((anchor = IdAnchor.get (r)) != null) {
    	    	if ((ancestryOp = super.AncestryOp (r)) != null)
    	    	    nodePatterns = new NodePatterns (r);
    	    	return;
    	    }
    	    if ((rootPattern = super.RootPattern (r)) == null)
    	        throw new IOException ("illegal ancestry pattern");
    	    nodePatterns = NodePatterns.get (r);
    	}
    	
    	boolean match (Element e, StyleSheet s)
    	{
	    if (anchor != null) {
// XXX match ancestry too
		if (!anchor.match (e, s))
		    return false;
		return nodePatterns == null || nodePatterns.match (e, s);
	    }
    	    if (rootPattern != null) {
    	    	if (e.getParentNode () != e.getOwnerDocument ())
    	    	    return false;
    	    	return nodePatterns == null || nodePatterns.match (e, s);
    	    }
    	    return nodePatterns.match (e, s);
    	}
    }
    
    // [4] (ElementPatterns (AncestryOp AttributePatterns)?)
    //		| AttributePattern
    static class NodePatterns extends XslPattern
    {
    	private ElementPatterns		elements;
    	// private AttributePattern	attributes;
    	
    	static private NodePatterns get (Reader r)
    	{
    	    try {
    	    	return new NodePatterns (r);
    	    } catch (IOException e) {
    	    	return null;
    	    }
    	}
    	
    	NodePatterns (Reader r) throws IOException
    	{
    	    elements = new ElementPatterns (r);
    	    
// XXX handle attribute patterns!
    	}
    	
    	boolean match (Element e, StyleSheet s)
    	{
    	    return elements.match (e, s);
    	}
    }
    
    // [5] ElementPattern (AncestryOp ElementPattern)*
    static class ElementPatterns extends XslPattern
    {
    	private ElementPattern		self;
    	private String			ancestryOp;
    	private ElementPatterns		next;
    	
    	ElementPatterns (Reader r) throws IOException
    	{
    	    self = new ElementPattern (r);
    	    if ((ancestryOp = super.AncestryOp (r)) != null)
    	        next = new ElementPatterns (r);
    	}
    	
    	boolean match (Element e, StyleSheet s)
    	{
    	    return (next == null)
    	        ? self.match (e, s)
    	        : ancestryMatch (e, s) != null;
    	}
   	
    	Element ancestryMatch (Element e, StyleSheet s)
    	{
    	    if (next == null)
    	        return self.match (e, s) ? e : null;
	    if ("/".equals (ancestryOp)) {
    	    	Element		match = next.ancestryMatch (e, s);
    	    	Element		parent;
    	    	
    	    	if (match == null
    	    		|| !(match.getParentNode () instanceof Element))
    	    	    return null;
    	    	    
    	    	parent = (Element) match.getParentNode ();
    	    	if (self.match (parent, s))
    	    	    return parent;
    	    	return null;
    	    }
// XXX System.out.println ("ignoring " + ancestryOp);
    	    return null;
    	}
    }
    
    // [6] "/" | "//" ... or null
    static String AncestryOp (Reader r) throws IOException
    {
    	if (RootPattern (r) == null)	// shared lines'o'code
    	    return null;
    	r.mark (1);
    	if (r.read () != '/') {
    	    r.reset ();
    	    return "/";
    	}
    	return "//";
    }
    
    // [7] ElementTypePattern ElementQualification?
    static class ElementPattern extends XslPattern
    {
    	private ElementTypePattern	elementType;
    	// private ElementQualification	elementQualification;
    	
    	ElementPattern (Reader r) throws IOException
    	{
    	    elementType = new ElementTypePattern (r);
    	    // elementQualification = ElementQualification.get (r);
    	}
    	
    	boolean match (Element e, StyleSheet s)
    	{
    	    if (!elementType.match (e, s))
    	        return false;
    	    // return elementQualification == null
    	    //		|| elementQualification.match (e, s);
    	    return true;
    	}
    }
    
    // Anchor
    // AbsoluteAnchor
    // RelativeAnchor
    // CurrentNodeAnchor
    // ParentAnchor
    // AncestorAnchor
    // IdAnchor

    static class IdAnchor extends XslPattern
    {
	private String		id;

	static IdAnchor get (Reader r)
	{
	    try {
		return new IdAnchor (r);
	    } catch (IOException e) {
		return null;
	    }
	}

    	IdAnchor (Reader r) throws IOException
    	{
	    if (!super.peek (r, "id\u0028"))	// left paren
		throw new IOException ("not an ID anchor");
// XXX NCName, then rparen \u0029
throw new RuntimeException ("absolute anchor parsing nyi");
    	}
    	
    	boolean match (Element e, StyleSheet s)
    	{
	    String	att = ((ElementNode)e).getIdAttributeName ();

	    // NOTE:  this won't support multiple ID attributes
	    // if the DTD defines the ID.

	    if (att == null) {
		att = s.getId (e);
		if (att == null)
		    return false;
	    } else
		att = e.getAttribute (att);
	    return att != null && id.equals (att);
    	}
    }

    
    // [15] "/"  ... or null
    static String RootPattern (Reader r) throws IOException
    {
    	r.mark (1);
    	if (r.read () != '/') {
    	    r.reset ();
    	    return null;
    	}
    	return "/";
    }
    
    // [16] OneElementTypePattern | AnyElementTypePattern
    static class ElementTypePattern extends XslPattern
    {
    	private String		element;
    	
    	ElementTypePattern (Reader r) throws IOException
    	{
    	    int			c;
    	    
    	    r.mark (1);
    	    if ((c = r.read ()) == '*')
    	        return;
    	    if (c == -1)
    	        throw new EOFException ("end of input");
    	    if (!(XmlChars.isLetter ((char)c) || c == '_')) {
    	        r.reset ();
    	        throw new IOException ("not an XML letter");
    	    }
    	    
    	    StringBuffer	buf = new StringBuffer ();
    	    boolean		sawColon = false;
    	    
    	    buf.append ((char) c);
    	    r.mark (1);
    	    while ((c = r.read ()) != -1) {
    	    	if (XmlChars.isNCNameChar ((char)c)) {
    	    	    buf.append ((char) c);
    	            r.mark (1);
    	    	    continue;
    	    	}
    	    	if (c == ':') {
    	    	    if (sawColon)
    	    	        throw new IOException ("illegal XML NCname");
    	    	    sawColon = true;
    	    	    buf.append ((char) c);
    	    	    c = r.read ();
    	    	    if (!XmlChars.isLetter ((char)c))
    	    	        throw new IOException (
    	    	        	"illegal XML NCname, not a Letter");
    	    	    buf.append ((char) c);
    	    	    r.mark (1);
    	    	    continue;
    	        }
    	        r.reset ();
    	        break;
    	    }
    	    element = buf.toString ();
    	}
    	
    	boolean match (Element e, StyleSheet s)
    	{
    	    return element == null || e.getNodeName ().equals (element);
    	}
    }

    static boolean peek (Reader r, String s)
    throws IOException
    {
	int 	len = s.length ();
	int	i;

	r.mark (len);
	for (i = 0; i < len; i++)
	    if (r.read () != s.charAt (i)) {
		r.reset ();
		return false;
	    }
	return true;
    }
}
