// SAX parser factory.
// No warranty; no copyright -- use this as you will.
// $Id: ParserFactory.java,v 1.1 1999/02/26 21:34:25 mode Exp $

package org.xml.sax.helpers;

import java.lang.ClassNotFoundException;
import java.lang.IllegalAccessException;
import java.lang.InstantiationException;
import java.lang.SecurityException;
import java.lang.ClassCastException;

import org.xml.sax.Parser;


/**
  * Java-specific class for dynamically loading SAX parsers.
  *
  * <p>This class is not part of the platform-independent definition
  * of SAX; it is an additional convenience class designed
  * specifically for Java XML application writers.  SAX applications
  * can use the static methods in this class to allocate a SAX parser
  * dynamically at run-time based either on the value of the
  * `org.xml.sax.parser' system property or on a string containing the class
  * name.</p>
  *
  * <p>Note that the application still requires an XML parser that
  * implements SAX.</p>
  *
  * <P><em>Note:  in this distribution, if you have not configured
  * your system properties or explicitly requested a given parser,
  * you will get the <code>com.sun.xml.parser.Parser</code> parser.
  * That is a fast parser which does not validate, but does process
  * all external entities and will report ignorable whitespace as such
  * in XML documents with DTDs.
  * </em>
  *
  * @author David Megginson (ak117@freenet.carleton.ca)
  * @version 1.0
  * @see org.xml.sax.Parser
  * @see java.lang.Class
  */
public class ParserFactory {

    // In case the user doesn't set the system property to
    // select the parser, we default to one.

    private static final String DEFAULT_PARSER =
	    // "com.jclark.xml.sax.Driver";
	    // "com.microstar.xml.SAXDriver";
	    "com.sun.xml.parser.Parser";


  /**
   * Private null constructor.
   */
  private ParserFactory ()
  {
  }


  /**
    * Create a new SAX parser using the `org.xml.sax.parser' system property.
    *
    * <p>The named class must exist and must implement the
    * org.xml.sax.Parser interface.</p>
    *
    * @exception java.lang.NullPointerException There is no value
    *            for the `org.xml.sax.parser' system property.
    * @exception java.lang.ClassNotFoundException The SAX parser
    *            class was not found (check your CLASSPATH).
    * @exception IllegalAccessException The SAX parser class was
    *            found, but you do not have permission to load
    *            it.
    * @exception InstantiationException The SAX parser class was
    *            found but could not be instantiated.
    * @exception java.lang.ClassCastException The SAX parser class
    *            was found and instantiated, but does not implement
    *            org.xml.sax.Parser.
    * @see #makeParser(java.lang.String)
    * @see org.xml.sax.Parser
    */
  public static Parser makeParser ()
    throws ClassNotFoundException,
           IllegalAccessException, 
           InstantiationException,
           NullPointerException,
           ClassCastException
  {
    String className = DEFAULT_PARSER;
    
    try {
	className = System.getProperty ("org.xml.sax.parser",
	    DEFAULT_PARSER);
    } catch (SecurityException x) {
	// e.g. in browser
    }
    return makeParser(className);
  }


  /**
    * Create a new SAX parser object using the class name provided.
    *
    * <p>The named class must exist and must implement the
    * org.xml.sax.Parser interface.</p>
    *
    * @param className A string containing the name of the
    *                  SAX parser class.
    * @exception java.lang.ClassNotFoundException The SAX parser
    *            class was not found (check your CLASSPATH).
    * @exception IllegalAccessException The SAX parser class was
    *            found, but you do not have permission to load
    *            it.
    * @exception InstantiationException The SAX parser class was
    *            found but could not be instantiated.
    * @exception java.lang.ClassCastException The SAX parser class
    *            was found and instantiated, but does not implement
    *            org.xml.sax.Parser.
    * @see #makeParser()
    * @see org.xml.sax.Parser
    */
  public static Parser makeParser (String className)
    throws ClassNotFoundException,
           IllegalAccessException, 
           InstantiationException,
           ClassCastException
  {
    return (Parser)(Class.forName(className).newInstance());
  }

}

