/*
 * @(#)StyleSheet.java	1.8 98/11/18
 * 
 * Copyright (c) 1998 Sun Microsystems, Inc. All Rights Reserved.
 * 
 * This software is the confidential and proprietary information of Sun
 * Microsystems, Inc. ("Confidential Information").  You shall not
 * disclose such Confidential Information and shall use it only in
 * accordance with the terms of the license agreement you entered into
 * with Sun.
 * 
 * SUN MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF THE
 * SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR
 * PURPOSE, OR NON-INFRINGEMENT. SUN SHALL NOT BE LIABLE FOR ANY DAMAGES
 * SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING OR DISTRIBUTING
 * THIS SOFTWARE OR ITS DERIVATIVES.
 */


package com.sun.xml.xsl;

import java.util.Vector;

import org.w3c.dom.*;

import com.sun.xml.tree.*;


/**
 * Instance of this class represent XSL stylesheets.  These get applied
 * to XML documents, and produce DOM trees.
 *
 * <P> <em>Note:</em> These documents are heavily modified as part of
 * XSL processing.  Don't use them except to apply them to other XML
 * documents!
 */
final public class StyleSheet extends XmlDocument
{
    // package private
    static final String	uri = "http://www.w3.org/TR/WD-xsl";
    
    private boolean	initialized;
    private Vector	templates;
    private Template	defaultTemplate;
    
    
    /**
     * Constructs a XSL stylesheet document.
     */
    public StyleSheet ()
    {
	setElementFactory (new EFactory ());
    }

    static class EFactory implements ElementFactory
    {

	public ElementEx createElementEx (String tag)
	{
	    return new ElementNode ();
	}

	public ElementEx createElementEx (String namespace, String tag)
	{
	    // processing within sheets is like nested templates
	    if (!uri.equals (namespace))
		return new Template ();

	    /* Top level stylesheet elements: */
	    if ("stylesheet".equals (tag))
		return new SheetRoot ();
	    if ("template".equals (tag))
		return new Template ();
	    if ("import".equals (tag))
		return new ExternSheet ();
	    if ("include".equals (tag))
		return new ExternSheet ();

	    // id
	    // strip-space
	    // preserve-space
	    // define-macro
	    // define-attribute-set
	    // define-constant
	    
	    /* XSL-specific nodes within templates */
	    if ("process-children".equals (tag))
		return new ProcessChildren ();
	    // process
	    // for-each
	    // value-of
	    // number
	    // choose
	    // if
	    // contents
	    // invoke
	    // text

	    return new ElementNode ();
	}
    }

    // XXX want:  void apply (input, output) so that output
    // nodes can be HTML nodes (etc).

    /**
     * Apply the stylesheet to the indicated document.
     *
     * <P><em>Note:</em> In the case of large input sources, the
     * DOM nodes in the input are expected to be self-expanding
     * in the sense that children may be produced on demand.  An
     * example would be a relational database query.  Every effort
     * will be made to minimize the amount of data that "must" be
     * stored in memory at one time, permitting large data sets to
     * be streamed efficiently.
     *
     * @param input the document which is processed, and which may
     *	be modified as part of the stylesheet processing.
     */
    public XmlDocument apply (XmlDocument input)
    {
    	XmlDocument	retval = new XmlDocument ();
    	ElementNode	root = (ElementNode) input.getDocumentElement ();
    	
    	// eventually, context will hold parameters etc
    	// or perhaps constants
    	XslContext	context = new XslContext ();
    	
    	if (!initialized)
    	    init();
    	
    	retval.appendChild (retval.createComment (
    		"\n\nCreated using an XSL stylesheet\nDo not edit!!\n\n"
    		));
    	    	
    	// walk tree ...
    	//	mark whitespace as ignorable, emitting if isIndentingResult
    	
    	findTemplate (root).process (root, retval, context);
    	return retval;
    }
    
    private void init ()
    {
    	// XSL Builder normalized and did includes, imports
    	
    	// strip whitespace ... {strip,preserve}-space
    	//	... also stylesheet's default-space attribute
    	//	... also xml:space
    	//	... and apply it to input nodes!!
    	
    	// prepare macros (define-macro)
    	
    	// prepare attribute sets (define-attribute-set) for 'use'
    	
    	// prepare constants (define-constant) for {constant(xxx)}
    	
    	// IMPLICIT:  default rule matching "* | /", process children
    	//	... can be overridden, only used if not overridden

    	defaultTemplate = (Template) createElementEx (uri, "template");
    	defaultTemplate.setAttribute ("xmlns", uri);
    	defaultTemplate.setAttribute ("matches", "*|/");
	defaultTemplate.startParse (null);
    	defaultTemplate.appendChild (
	    createElementEx (uri, "process-children"));
    	
    	initialized = true;
    }
    
    // package private
    Template findTemplate (ElementNode e)
    {
    	SheetRoot	root = (SheetRoot) getDocumentElement ();
    	
    	for (int i = root.getLength () - 1; i >= 0; --i) {
    	    ElementNode	child = (ElementNode) root.item (i);
	    Template	template;
    	    
    	    if (!"template".equals (child.getLocalName ()))
    	        continue;
	    template = (Template) child;
    	    if (template.match (e))
    	    	return template;
    	}
    	
    	return defaultTemplate;
    }

    /**
     * Returns the ID assigned to the element by "id" elements,
     * or null if none was assigned.
     */
    // package private
    String getId (Element e)
    {
    	SheetRoot	root = (SheetRoot) getDocumentElement ();
    	
    	for (int i = root.getLength () - 1; i >= 0; --i) {
    	    Node	child = root.item (i);
	    ElementNode	element;
	    String	tmp;
    	    
	    element = (ElementNode) child;
    	    if (!"id".equals (element.getLocalName ())
		    || !StyleSheet.uri.equals (element.getNamespace ()))
    	        continue;

	    // XXX how does XSL deal with prefixes which may
	    // differ in the style sheet and in the document?

	    tmp = element.getAttribute (StyleSheet.uri, "element");
	    if (tmp != null && !tmp.equals (e.getNodeName ()))
		continue;
	    
	    tmp = element.getAttribute (StyleSheet.uri, "attribute");
	    if (tmp == null)
		throw new IllegalArgumentException (
		    "illegal XSL id, no 'attribute'");
	    
	    tmp = e.getAttribute (tmp);
	    if (tmp != null)
		return tmp;
    	}
    	return null;
    }
}
