/*
 * @(#)Base64Encoder.java	1.1 98/04/29
 * 
 * Copyright (c) 1998 Sun Microsystems, Inc. All Rights Reserved.
 * 
 * This software is the confidential and proprietary information of Sun
 * Microsystems, Inc. ("Confidential Information").  You shall not
 * disclose such Confidential Information and shall use it only in
 * accordance with the terms of the license agreement you entered into
 * with Sun.
 * 
 * SUN MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF THE
 * SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR
 * PURPOSE, OR NON-INFRINGEMENT. SUN SHALL NOT BE LIABLE FOR ANY DAMAGES
 * SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING OR DISTRIBUTING
 * THIS SOFTWARE OR ITS DERIVATIVES.
 */

package com.sun.xml.io;


import java.io.*;

/**
 * This output stream encodes its input data using the "base64" encoding,
 * before sending it to the <code>Writer</code> used in its construction.
 * Base64 encoding, specified in RFC 1521, is a way to encode binary data
 * for embedding in text formats such as MIME messages or XML.  The output
 * is a series of lines of encoded data.
 * 
 * <P> The data will in some cases be self-terminating:  unless the length
 * of input is an exact multiple of three bytes long, the encoding will mark
 * its end internally.  Otherwise, applications encoding data should mark
 * the end of the data in some way that the receiver can detect it.
 *
 * @see Base64Decoder
 *
 * @version 1.1
 * @author David Brownell
 */
public class Base64Encoder extends OutputStream {
    // where the data goes, and line length info
    private Writer	out;
    private int		count;

    // the buffered data
    private byte	buffer [] = new byte [3];
    private int		offset = 0;

    // the encoded data
    private char	data [] = new char [4];
    private boolean	eof = false;


    // line length MAX of 76 chars per rfc1521
    // ... or of 64 per the older rfc1421

    /**
     * Constructs an output stream which emits Base64 encoded data
     * on the given text stream.  This stream must be closed when
     * the last binary data is written on it.
     */
    public Base64Encoder (Writer out)
    {
	this.out = out;
    }

    static private final char encoding [] = {
	'A', 'B', 'C', 'D', 'E', 'F', 'G', 'H',
	'I', 'J', 'K', 'L', 'M', 'N', 'O', 'P',
	'Q', 'R', 'S', 'T', 'U', 'V', 'W', 'X',
	'Y', 'Z', 'a', 'b', 'c', 'd', 'e', 'f',
	'g', 'h', 'i', 'j', 'k', 'l', 'm', 'n',
	'o', 'p', 'q', 'r', 's', 't', 'u', 'v',
	'w', 'x', 'y', 'z', '0', '1', '2', '3',
	'4', '5', '6', '7', '8', '9', '+', '/'
    };

    // take three more bytes of data, encode it, write it
    private void flushData () throws IOException
    {
	byte		b1 = 0, b2 = 0, b3 = 0;
	int		tmp;

	// "offset" == index of next character (3 == full buffer)
	if (eof || offset == 0)
	    return;

	// per rfc 1521, we limit line lengths to 76
	if (count > 75) {
	    out.write ('\n');
	    count = 4;
	} else
	    count += 4;

	// all data bytes are split between two characters

	// first character we can always fill (except at eof)
	b1 = buffer [0];
	data [0] = encoding [0x3f & (
	    b1 >> 2
	    )];

	// second char has data from first and second byte
	if (offset == 1) {
	    data [1] = encoding [ 0x3f & (
		(b1 << 4)
		)];
	    data [2] = '=';
	    data [3] = '=';
	    eof = true;
	    out.write (data, 0, 4);
	    return;
	}
	b2 = buffer [1];
	data [1] = encoding [ 0x3f & (
	    (b1 << 4) + ((b2 >> 4) & 0xf)
	    )];

	// third char has data from second and third byte
	// fourth char has data only from third byte
	if (offset == 2) {
	    data [2] = encoding [ 0x3f & (
		b2 << 2
		)];
	    data [3] = '=';
	    eof = true;
	} else {
	    b3 = buffer [2];
	    data [2] = encoding [ 0x3f & (
		(b2 << 2) + ((b3 >> 6) & 0x3)
		)];
	    data [3] = encoding [ 0x3f & (
		b3 & 0x3f
		)];
	}
	out.write (data, 0, 4);
	offset = 0;
    }

    /**
     * Writes a single byte of binary data.  This will often be
     * stored in a small internal buffer.
     */
    public void write (int b) throws IOException
    {
	if (eof)
	    throw new EOFException ("closed");
	buffer [offset++] = (byte) b;
	if (offset == 3)
	    flushData ();
    }

    /**
     * Closes this stream, and flushes any buffered data.
     */
    public void close () throws IOException
    {
	flushData ();
	eof = true;
	buffer = null;
	data = null;
	out = null;
    }

    /*
    public static void main (String argv [])
    {
	try {
	    Writer 		w = new OutputStreamWriter (System.out);
	    OutputStream	out = new Base64Encoder (w);
	    PrintStream		ps = new PrintStream (out);

	    ps.println ("Hello, world??");
	    //ps.print ("admin");
	    ps.close ();
	    out.close ();
	    w.write ("\n");
	    w.flush ();

	} catch (Throwable t) {
	    t.printStackTrace ();
	}
    }
    */
}
